// API Service for 39Ricambi
const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:3001/api';

// Admin credentials (stored in memory only)
let adminCredentials: { username: string; password: string } | null = null;

export function setAdminCredentials(username: string, password: string) {
  adminCredentials = { username, password };
}

export function getAdminCredentials() {
  return adminCredentials;
}

export function clearAdminCredentials() {
  adminCredentials = null;
}

// API helper functions
async function fetchAPI(endpoint: string, options: RequestInit = {}) {
  const url = `${API_URL}${endpoint}`;
  
  const response = await fetch(url, {
    ...options,
    headers: {
      'Content-Type': 'application/json',
      ...options.headers,
    },
  });
  
  if (!response.ok) {
    const error = await response.json().catch(() => ({ message: 'Unknown error' }));
    throw new Error(error.message || `HTTP ${response.status}`);
  }
  
  return response.json();
}

// ==================== CREDENTIALS API ====================

export async function getCredentials() {
  return fetchAPI('/credentials');
}

export async function updateCredentials(username: string, password: string) {
  const adminCreds = getAdminCredentials();
  if (!adminCreds) {
    throw new Error('Admin not authenticated');
  }
  
  return fetchAPI('/credentials', {
    method: 'POST',
    body: JSON.stringify({
      username,
      password,
      adminUsername: adminCreds.username,
      adminPassword: adminCreds.password,
    }),
  });
}

// ==================== AUTH API ====================

export async function loginAPI(username: string, password: string) {
  return fetchAPI('/login', {
    method: 'POST',
    body: JSON.stringify({ username, password }),
  });
}

// ==================== PRODUCTS API ====================

export async function getProducts() {
  return fetchAPI('/products');
}

export async function addProductAPI(product: { id: string; category: string; model: string; price: number }) {
  const adminCreds = getAdminCredentials();
  if (!adminCreds) {
    throw new Error('Admin not authenticated');
  }
  
  return fetchAPI('/products', {
    method: 'POST',
    body: JSON.stringify({
      ...product,
      adminUsername: adminCreds.username,
      adminPassword: adminCreds.password,
    }),
  });
}

export async function updateProductAPI(id: string, updates: { category: string; model: string; price: number }) {
  const adminCreds = getAdminCredentials();
  if (!adminCreds) {
    throw new Error('Admin not authenticated');
  }
  
  return fetchAPI(`/products/${id}`, {
    method: 'PUT',
    body: JSON.stringify({
      ...updates,
      adminUsername: adminCreds.username,
      adminPassword: adminCreds.password,
    }),
  });
}

export async function deleteProductAPI(id: string) {
  const adminCreds = getAdminCredentials();
  if (!adminCreds) {
    throw new Error('Admin not authenticated');
  }
  
  return fetchAPI(`/products/${id}`, {
    method: 'DELETE',
    body: JSON.stringify({
      adminUsername: adminCreds.username,
      adminPassword: adminCreds.password,
    }),
  });
}

export async function importProductsAPI(products: Array<{ id: string; category: string; model: string; price: number }>) {
  const adminCreds = getAdminCredentials();
  if (!adminCreds) {
    throw new Error('Admin not authenticated');
  }
  
  return fetchAPI('/products/import', {
    method: 'POST',
    body: JSON.stringify({
      products,
      adminUsername: adminCreds.username,
      adminPassword: adminCreds.password,
    }),
  });
}

// Health check
export async function checkHealth() {
  return fetchAPI('/health');
}
