import { useState, useMemo } from 'react';
import { useAuth } from '@/context/AuthContext';
import { useInventory } from '@/context/InventoryContext';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Card, CardContent, CardHeader } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { 
  Search, 
  LogOut, 
  Smartphone, 
  Filter,
  ChevronDown,
  ChevronUp,
  Package,
  MessageCircle
} from 'lucide-react';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';
import {
  Collapsible,
  CollapsibleContent,
  CollapsibleTrigger,
} from '@/components/ui/collapsible';

// Category popularity order (most popular first)
const CATEGORY_ORDER: Record<string, number> = {
  'iPhone': 1,
  'Samsung S Series': 2,
  'Samsung A Series': 3,
  'Samsung Note': 4,
  'Samsung Z Fold': 5,
  'Samsung Z Flip': 6,
  'Samsung M Series': 7,
  'Samsung J Series': 8,
  'Samsung Tablet': 9,
  'iPad Pro': 10,
  'Apple Watch': 11,
  'Google Pixel': 12,
  'Xiaomi': 13,
  'Redmi Note': 14,
  'Redmi': 15,
  'Poco': 16,
  'OnePlus': 17,
  'Realme': 18,
  'Oppo Reno': 19,
  'Oppo A Series': 20,
  'Vivo': 21,
  'Lenovo Tablet': 22,
};

export function UserView() {
  const { logout } = useAuth();
  const { products, getCategories } = useInventory();
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedCategory, setSelectedCategory] = useState<string>('all');
  
  // Get all categories initially for expanded state
  const categories = getCategories();
  // Initially expand all categories
  const [expandedCategories, setExpandedCategories] = useState<Set<string>>(new Set(categories));

  // Sort categories by popularity
  const sortedCategories = useMemo(() => {
    return [...categories].sort((a, b) => {
      const orderA = CATEGORY_ORDER[a] || 999;
      const orderB = CATEGORY_ORDER[b] || 999;
      return orderA - orderB;
    });
  }, [categories]);

  // Group products by category with sorting
  const groupedProducts = useMemo(() => {
    const filtered = products.filter((product) => {
      const matchesSearch = product.model.toLowerCase().includes(searchQuery.toLowerCase()) ||
        product.category.toLowerCase().includes(searchQuery.toLowerCase());
      const matchesCategory = selectedCategory === 'all' || product.category === selectedCategory;
      return matchesSearch && matchesCategory;
    });

    const grouped: Record<string, typeof products> = {};
    filtered.forEach((product) => {
      if (!grouped[product.category]) {
        grouped[product.category] = [];
      }
      grouped[product.category].push(product);
    });

    // Sort the grouped entries by category popularity
    const sortedGrouped: Record<string, typeof products> = {};
    sortedCategories.forEach((cat) => {
      if (grouped[cat]) {
        sortedGrouped[cat] = grouped[cat];
      }
    });

    return sortedGrouped;
  }, [products, searchQuery, selectedCategory, sortedCategories]);

  const toggleCategory = (category: string) => {
    const newExpanded = new Set(expandedCategories);
    if (newExpanded.has(category)) {
      newExpanded.delete(category);
    } else {
      newExpanded.add(category);
    }
    setExpandedCategories(newExpanded);
  };

  const expandAll = () => {
    setExpandedCategories(new Set(Object.keys(groupedProducts)));
  };

  const collapseAll = () => {
    setExpandedCategories(new Set());
  };

  return (
    <div className="min-h-screen bg-slate-50">
      {/* Header */}
      <header className="sticky top-0 z-50 bg-white border-b border-slate-200 shadow-sm">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex items-center justify-between h-16">
            <div className="flex items-center gap-3">
              <div className="w-10 h-10 rounded-xl bg-gradient-to-br from-blue-500 to-purple-600 flex items-center justify-center">
                <Smartphone className="w-5 h-5 text-white" />
              </div>
              <div className="relative">
                <h1 className="text-xl font-bold text-slate-900">39Ricambi</h1>
                <span className="absolute -bottom-1 -right-6 text-[10px] text-slate-500">(b2b)</span>
              </div>
            </div>
            <Button
              variant="outline"
              size="sm"
              onClick={logout}
              className="gap-2 text-slate-600 hover:text-red-600 hover:border-red-200"
            >
              <LogOut className="w-4 h-4" />
              Logout
            </Button>
          </div>
        </div>
      </header>

      {/* Main Content */}
      <main className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Stats - Only Total Products and Categories */}
        <div className="grid grid-cols-2 gap-4 mb-8">
          <Card className="bg-gradient-to-br from-blue-500 to-blue-600 border-0 text-white">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-blue-100 text-sm">Total Products</p>
                  <p className="text-2xl font-bold">{products.length}</p>
                </div>
                <Package className="w-8 h-8 text-blue-200" />
              </div>
            </CardContent>
          </Card>
          <Card className="bg-gradient-to-br from-purple-500 to-purple-600 border-0 text-white">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-purple-100 text-sm">Categories</p>
                  <p className="text-2xl font-bold">{categories.length}</p>
                </div>
                <Filter className="w-8 h-8 text-purple-200" />
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Search and Filter */}
        <Card className="mb-6 shadow-sm">
          <CardContent className="p-4">
            <div className="flex flex-col md:flex-row gap-4">
              <div className="relative flex-1">
                <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-slate-400" />
                <Input
                  placeholder="Search by model or category..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="pl-10"
                />
              </div>
              <div className="flex gap-2">
                <Select value={selectedCategory} onValueChange={setSelectedCategory}>
                  <SelectTrigger className="w-[200px]">
                    <SelectValue placeholder="All Categories" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Categories</SelectItem>
                    {sortedCategories.map((cat) => (
                      <SelectItem key={cat} value={cat}>
                        {cat}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                <Button variant="outline" onClick={expandAll} size="sm">
                  Expand All
                </Button>
                <Button variant="outline" onClick={collapseAll} size="sm">
                  Collapse All
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Product List */}
        <div className="space-y-4">
          {Object.entries(groupedProducts).length === 0 ? (
            <Card className="p-12 text-center">
              <Package className="w-12 h-12 text-slate-300 mx-auto mb-4" />
              <p className="text-slate-500 text-lg">No products found</p>
              <p className="text-slate-400 text-sm">Try adjusting your search or filter</p>
            </Card>
          ) : (
            Object.entries(groupedProducts).map(([category, items]) => (
              <Collapsible
                key={category}
                open={expandedCategories.has(category)}
                onOpenChange={() => toggleCategory(category)}
              >
                <Card className="overflow-hidden shadow-sm hover:shadow-md transition-shadow">
                  <CollapsibleTrigger asChild>
                    <CardHeader className="bg-gradient-to-r from-slate-50 to-white cursor-pointer py-4">
                      <div className="flex items-center justify-between">
                        <div className="flex items-center gap-3">
                          <Badge variant="secondary" className="text-sm font-semibold px-3 py-1">
                            {category}
                          </Badge>
                          <span className="text-slate-500 text-sm">({items.length} items)</span>
                        </div>
                        {expandedCategories.has(category) ? (
                          <ChevronUp className="w-5 h-5 text-slate-400" />
                        ) : (
                          <ChevronDown className="w-5 h-5 text-slate-400" />
                        )}
                      </div>
                    </CardHeader>
                  </CollapsibleTrigger>
                  <CollapsibleContent>
                    <CardContent className="p-0">
                      <div className="divide-y divide-slate-100">
                        {items.map((product) => (
                          <div
                            key={product.id}
                            className="flex items-center justify-between p-4 hover:bg-slate-50 transition-colors"
                          >
                            <div className="flex-1">
                              <p className="font-medium text-slate-900">{product.model}</p>
                            </div>
                            <a
                              href="https://wa.me/+393508181116"
                              target="_blank"
                              rel="noopener noreferrer"
                              className="inline-flex items-center gap-2 px-4 py-2 bg-green-500 hover:bg-green-600 text-white rounded-lg font-bold transition-colors"
                            >
                              <MessageCircle className="w-4 h-4" />
                              €{Math.floor(product.price)}.00
                            </a>
                          </div>
                        ))}
                      </div>
                    </CardContent>
                  </CollapsibleContent>
                </Card>
              </Collapsible>
            ))
          )}
        </div>
      </main>
    </div>
  );
}
