import { useState, useMemo, useRef } from 'react';
import { useAuth } from '@/context/AuthContext';
import { useInventory } from '@/context/InventoryContext';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { 
  Search, 
  LogOut, 
  Smartphone, 
  Plus, 
  Edit2, 
  Trash2, 
  X,
  Save,
  Package,
  Filter,
  TrendingUp,
  Users,
  Shield,
  MessageCircle,
  Key,
  RefreshCw,
  Copy,
  Check,
  Upload,
  Download
} from 'lucide-react';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from '@/components/ui/dialog';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from '@/components/ui/alert-dialog';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import type { Product } from '@/types';
import { importProductsAPI } from '@/services/api';

// Category popularity order (most popular first)
const CATEGORY_ORDER: Record<string, number> = {
  'iPhone': 1,
  'Samsung S Series': 2,
  'Samsung A Series': 3,
  'Samsung Note': 4,
  'Samsung Z Fold': 5,
  'Samsung Z Flip': 6,
  'Samsung M Series': 7,
  'Samsung J Series': 8,
  'Samsung Tablet': 9,
  'iPad Pro': 10,
  'Apple Watch': 11,
  'Google Pixel': 12,
  'Xiaomi': 13,
  'Redmi Note': 14,
  'Redmi': 15,
  'Poco': 16,
  'OnePlus': 17,
  'Realme': 18,
  'Oppo Reno': 19,
  'Oppo A Series': 20,
  'Vivo': 21,
  'Lenovo Tablet': 22,
};

export function AdminDashboard() {
  const { logout, customerCredentials, updateCustomerCredentials } = useAuth();
  const { products, addProduct, updateProduct, deleteProduct, getCategories } = useInventory();
  
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedCategory, setSelectedCategory] = useState<string>('all');
  
  // Modal states
  const [isAddModalOpen, setIsAddModalOpen] = useState(false);
  const [isEditModalOpen, setIsEditModalOpen] = useState(false);
  const [isDeleteDialogOpen, setIsDeleteDialogOpen] = useState(false);
  const [isCredentialModalOpen, setIsCredentialModalOpen] = useState(false);
  const [isImportModalOpen, setIsImportModalOpen] = useState(false);
  const [selectedProduct, setSelectedProduct] = useState<Product | null>(null);
  
  // Form states
  const [formData, setFormData] = useState({
    category: '',
    model: '',
    price: '',
  });
  const [newCategory, setNewCategory] = useState('');
  const [useNewCategory, setUseNewCategory] = useState(false);

  // Credential generator states
  const [customUsername, setCustomUsername] = useState('');
  const [customPassword, setCustomPassword] = useState('');
  const [generatedCredentials, setGeneratedCredentials] = useState<{username: string, password: string} | null>(null);
  const [copied, setCopied] = useState(false);

  // Import states
  const [importError, setImportError] = useState('');
  const [importSuccess, setImportSuccess] = useState('');
  const fileInputRef = useRef<HTMLInputElement>(null);

  const categories = getCategories();

  // Sort categories by popularity
  const sortedCategories = useMemo(() => {
    return [...categories].sort((a, b) => {
      const orderA = CATEGORY_ORDER[a] || 999;
      const orderB = CATEGORY_ORDER[b] || 999;
      return orderA - orderB;
    });
  }, [categories]);

  // Filter products
  const filteredProducts = useMemo(() => {
    return products.filter((product) => {
      const matchesSearch = product.model.toLowerCase().includes(searchQuery.toLowerCase()) ||
        product.category.toLowerCase().includes(searchQuery.toLowerCase());
      const matchesCategory = selectedCategory === 'all' || product.category === selectedCategory;
      return matchesSearch && matchesCategory;
    });
  }, [products, searchQuery, selectedCategory]);

  // Stats
  const stats = useMemo(() => ({
    totalProducts: products.length,
    totalCategories: categories.length,
    avgPrice: products.reduce((acc, p) => acc + p.price, 0) / products.length || 0,
    totalValue: products.reduce((acc, p) => acc + p.price, 0),
  }), [products, categories]);

  const resetForm = () => {
    setFormData({ category: '', model: '', price: '' });
    setNewCategory('');
    setUseNewCategory(false);
  };

  const handleAdd = () => {
    const category = useNewCategory ? newCategory : formData.category;
    if (!category || !formData.model || !formData.price) return;

    addProduct({
      category,
      model: formData.model,
      price: parseFloat(formData.price),
    });
    
    resetForm();
    setIsAddModalOpen(false);
  };

  const handleEdit = () => {
    if (!selectedProduct) return;
    
    const category = useNewCategory ? newCategory : formData.category;
    if (!category || !formData.model || !formData.price) return;

    updateProduct(selectedProduct.id, {
      category,
      model: formData.model,
      price: parseFloat(formData.price),
    });
    
    resetForm();
    setIsEditModalOpen(false);
    setSelectedProduct(null);
  };

  const handleDelete = () => {
    if (selectedProduct) {
      deleteProduct(selectedProduct.id);
      setIsDeleteDialogOpen(false);
      setSelectedProduct(null);
    }
  };

  const openEditModal = (product: Product) => {
    setSelectedProduct(product);
    setFormData({
      category: product.category,
      model: product.model,
      price: product.price.toString(),
    });
    setUseNewCategory(false);
    setNewCategory('');
    setIsEditModalOpen(true);
  };

  const openDeleteDialog = (product: Product) => {
    setSelectedProduct(product);
    setIsDeleteDialogOpen(true);
  };

  // Credential generator functions
  const generateRandomCredentials = () => {
    const randomStr = Math.random().toString(36).substring(2, 8).toUpperCase();
    const username = `CLIENTE_${randomStr}`;
    const password = Math.random().toString(36).substring(2, 10);
    setGeneratedCredentials({ username, password });
  };

  const generateCustomCredentials = () => {
    if (customUsername && customPassword) {
      setGeneratedCredentials({ username: customUsername, password: customPassword });
    }
  };

  const saveCredentials = async () => {
    if (generatedCredentials) {
      try {
        await updateCustomerCredentials(generatedCredentials.username, generatedCredentials.password);
        setIsCredentialModalOpen(false);
        setGeneratedCredentials(null);
        setCustomUsername('');
        setCustomPassword('');
      } catch (err) {
        setImportError('Failed to save credentials. Please try again.');
      }
    }
  };

  const copyToClipboard = () => {
    if (generatedCredentials) {
      navigator.clipboard.writeText(`Username: ${generatedCredentials.username}\nPassword: ${generatedCredentials.password}`);
      setCopied(true);
      setTimeout(() => setCopied(false), 2000);
    }
  };

  const openCredentialModal = () => {
    setGeneratedCredentials(null);
    setCustomUsername('');
    setCustomPassword('');
    setCopied(false);
    setIsCredentialModalOpen(true);
  };

  // Export CSV function
  const exportCSV = () => {
    // Create CSV content
    const headers = ['Category', 'Models', 'Price (€)'];
    const rows = products.map(p => [
      `"${p.category.replace(/"/g, '""')}"`,
      `"${p.model.replace(/"/g, '""')}"`,
      Math.floor(p.price)
    ]);
    
    const csvContent = [
      headers.join(','),
      ...rows.map(row => row.join(','))
    ].join('\n');

    // Create and download file
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', `39ricambi_inventory_${new Date().toISOString().split('T')[0]}.csv`);
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  };

  // Import CSV function
  const handleFileUpload = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) return;

    const reader = new FileReader();
    reader.onload = async (e) => {
      try {
        const content = e.target?.result as string;
        const lines = content.split('\n').filter(line => line.trim());
        
        // Skip header
        const dataLines = lines.slice(1);
        
        const productsToImport: Array<{ id: string; category: string; model: string; price: number }> = [];
        
        dataLines.forEach(line => {
          // Parse CSV line (handle quoted fields)
          const matches = line.match(/("[^"]*"|[^,]*)/g);
          if (matches && matches.length >= 3) {
            const category = matches[0].replace(/^"|"$/g, '').replace(/""/g, '"').trim();
            const model = matches[1].replace(/^"|"$/g, '').replace(/""/g, '"').trim();
            const priceStr = matches[2].trim();
            
            if (category && model && priceStr) {
              const price = parseFloat(priceStr);
              if (!isNaN(price)) {
                productsToImport.push({
                  id: `prod_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
                  category,
                  model,
                  price
                });
              }
            }
          }
        });

        if (productsToImport.length > 0) {
          const response = await importProductsAPI(productsToImport);
          setImportSuccess(response.message || `Successfully imported ${productsToImport.length} products!`);
          setImportError('');
          setTimeout(() => {
            setIsImportModalOpen(false);
            setImportSuccess('');
          }, 2000);
        } else {
          setImportError('No valid products found in CSV file.');
        }
      } catch (error) {
        console.error('Import error:', error);
        setImportError('Error importing CSV file. Please check the format and try again.');
        setImportSuccess('');
      }
    };
    reader.readAsText(file);
  };

  const openImportModal = () => {
    setImportError('');
    setImportSuccess('');
    setIsImportModalOpen(true);
  };

  return (
    <div className="min-h-screen bg-slate-50">
      {/* Header */}
      <header className="sticky top-0 z-50 bg-white border-b border-slate-200 shadow-sm">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex items-center justify-between h-16">
            <div className="flex items-center gap-3">
              <div className="w-10 h-10 rounded-xl bg-gradient-to-br from-blue-500 to-purple-600 flex items-center justify-center">
                <Smartphone className="w-5 h-5 text-white" />
              </div>
              <div className="relative">
                <h1 className="text-xl font-bold text-slate-900">39Ricambi</h1>
                <span className="absolute -bottom-1 -right-6 text-[10px] text-slate-500">(b2b)</span>
              </div>
              <div className="flex items-center gap-2 ml-4">
                <Badge variant="default" className="bg-purple-600 text-xs">
                  <Shield className="w-3 h-3 mr-1" />
                  Admin
                </Badge>
              </div>
            </div>
            <Button
              variant="outline"
              size="sm"
              onClick={logout}
              className="gap-2 text-slate-600 hover:text-red-600 hover:border-red-200"
            >
              <LogOut className="w-4 h-4" />
              Logout
            </Button>
          </div>
        </div>
      </header>

      {/* Main Content */}
      <main className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Stats */}
        <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-8">
          <Card className="bg-gradient-to-br from-blue-500 to-blue-600 border-0 text-white">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-blue-100 text-sm">Total Products</p>
                  <p className="text-2xl font-bold">{stats.totalProducts}</p>
                </div>
                <Package className="w-8 h-8 text-blue-200" />
              </div>
            </CardContent>
          </Card>
          <Card className="bg-gradient-to-br from-purple-500 to-purple-600 border-0 text-white">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-purple-100 text-sm">Categories</p>
                  <p className="text-2xl font-bold">{stats.totalCategories}</p>
                </div>
                <Filter className="w-8 h-8 text-purple-200" />
              </div>
            </CardContent>
          </Card>
          <Card className="bg-gradient-to-br from-emerald-500 to-emerald-600 border-0 text-white">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-emerald-100 text-sm">Avg Price</p>
                  <p className="text-2xl font-bold">€{Math.floor(stats.avgPrice)}.00</p>
                </div>
                <TrendingUp className="w-8 h-8 text-emerald-200" />
              </div>
            </CardContent>
          </Card>
          <Card className="bg-gradient-to-br from-orange-500 to-orange-600 border-0 text-white">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-orange-100 text-sm">Total Value</p>
                  <p className="text-lg font-bold">€{stats.totalValue.toFixed(0)}</p>
                </div>
                <Users className="w-8 h-8 text-orange-200" />
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Search and Actions */}
        <Card className="mb-6 shadow-sm">
          <CardContent className="p-4">
            <div className="flex flex-col md:flex-row gap-4">
              <div className="relative flex-1">
                <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-slate-400" />
                <Input
                  placeholder="Search by model or category..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="pl-10"
                />
              </div>
              <div className="flex gap-2 flex-wrap">
                <Select value={selectedCategory} onValueChange={setSelectedCategory}>
                  <SelectTrigger className="w-[200px]">
                    <SelectValue placeholder="All Categories" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Categories</SelectItem>
                    {sortedCategories.map((cat) => (
                      <SelectItem key={cat} value={cat}>
                        {cat}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                <Button 
                  onClick={() => {
                    resetForm();
                    setIsAddModalOpen(true);
                  }}
                  className="bg-blue-600 hover:bg-blue-700"
                >
                  <Plus className="w-4 h-4 mr-2" />
                  Add Product
                </Button>
                <Button 
                  onClick={openCredentialModal}
                  className="bg-purple-600 hover:bg-purple-700"
                >
                  <Key className="w-4 h-4 mr-2" />
                  Credentials
                </Button>
                <Button 
                  onClick={exportCSV}
                  variant="outline"
                  className="border-green-500 text-green-600 hover:bg-green-50"
                >
                  <Download className="w-4 h-4 mr-2" />
                  Export CSV
                </Button>
                <Button 
                  onClick={openImportModal}
                  variant="outline"
                  className="border-orange-500 text-orange-600 hover:bg-orange-50"
                >
                  <Upload className="w-4 h-4 mr-2" />
                  Import CSV
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Products Table */}
        <Card className="shadow-sm">
          <CardHeader>
            <CardTitle className="text-lg">Inventory Management</CardTitle>
          </CardHeader>
          <CardContent className="p-0">
            <div className="overflow-x-auto">
              <Table>
                <TableHeader>
                  <TableRow className="bg-slate-50">
                    <TableHead className="font-semibold">Category</TableHead>
                    <TableHead className="font-semibold">Model</TableHead>
                    <TableHead className="font-semibold text-right">Price (€)</TableHead>
                    <TableHead className="font-semibold text-right">Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredProducts.length === 0 ? (
                    <TableRow>
                      <TableCell colSpan={4} className="text-center py-12">
                        <Package className="w-12 h-12 text-slate-300 mx-auto mb-4" />
                        <p className="text-slate-500">No products found</p>
                      </TableCell>
                    </TableRow>
                  ) : (
                    filteredProducts.map((product) => (
                      <TableRow key={product.id} className="hover:bg-slate-50">
                        <TableCell>
                          <Badge variant="secondary">{product.category}</Badge>
                        </TableCell>
                        <TableCell className="font-medium">{product.model}</TableCell>
                        <TableCell className="text-right">
                          <a
                            href="https://wa.me/+393508181116"
                            target="_blank"
                            rel="noopener noreferrer"
                            className="inline-flex items-center gap-2 px-3 py-1 bg-green-500 hover:bg-green-600 text-white rounded-lg font-bold transition-colors"
                          >
                            <MessageCircle className="w-4 h-4" />
                            €{Math.floor(product.price)}.00
                          </a>
                        </TableCell>
                        <TableCell className="text-right">
                          <div className="flex justify-end gap-2">
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() => openEditModal(product)}
                              className="text-blue-600 hover:bg-blue-50"
                            >
                              <Edit2 className="w-4 h-4" />
                            </Button>
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() => openDeleteDialog(product)}
                              className="text-red-600 hover:bg-red-50"
                            >
                              <Trash2 className="w-4 h-4" />
                            </Button>
                          </div>
                        </TableCell>
                      </TableRow>
                    ))
                  )}
                </TableBody>
              </Table>
            </div>
          </CardContent>
        </Card>
      </main>

      {/* Add Product Modal */}
      <Dialog open={isAddModalOpen} onOpenChange={setIsAddModalOpen}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle>Add New Product</DialogTitle>
            <DialogDescription>
              Add a new phone screen to the inventory.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label>Category</Label>
              {!useNewCategory ? (
                <Select 
                  value={formData.category} 
                  onValueChange={(value) => setFormData({ ...formData, category: value })}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Select category" />
                  </SelectTrigger>
                  <SelectContent>
                    {sortedCategories.map((cat) => (
                      <SelectItem key={cat} value={cat}>{cat}</SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              ) : (
                <Input
                  placeholder="Enter new category"
                  value={newCategory}
                  onChange={(e) => setNewCategory(e.target.value)}
                />
              )}
              <div className="flex items-center gap-2 mt-2">
                <input
                  type="checkbox"
                  id="newCat"
                  checked={useNewCategory}
                  onChange={(e) => setUseNewCategory(e.target.checked)}
                  className="rounded border-slate-300"
                />
                <label htmlFor="newCat" className="text-sm text-slate-600">Create new category</label>
              </div>
            </div>
            <div className="space-y-2">
              <Label>Model</Label>
              <Input
                placeholder="Enter model name"
                value={formData.model}
                onChange={(e) => setFormData({ ...formData, model: e.target.value })}
              />
            </div>
            <div className="space-y-2">
              <Label>Price (€)</Label>
              <Input
                type="number"
                step="0.01"
                placeholder="0.00"
                value={formData.price}
                onChange={(e) => setFormData({ ...formData, price: e.target.value })}
              />
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setIsAddModalOpen(false)}>
              <X className="w-4 h-4 mr-2" />
              Cancel
            </Button>
            <Button onClick={handleAdd} className="bg-blue-600 hover:bg-blue-700">
              <Save className="w-4 h-4 mr-2" />
              Add Product
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Edit Product Modal */}
      <Dialog open={isEditModalOpen} onOpenChange={setIsEditModalOpen}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle>Edit Product</DialogTitle>
            <DialogDescription>
              Update the product details.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label>Category</Label>
              {!useNewCategory ? (
                <Select 
                  value={formData.category} 
                  onValueChange={(value) => setFormData({ ...formData, category: value })}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Select category" />
                  </SelectTrigger>
                  <SelectContent>
                    {sortedCategories.map((cat) => (
                      <SelectItem key={cat} value={cat}>{cat}</SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              ) : (
                <Input
                  placeholder="Enter new category"
                  value={newCategory}
                  onChange={(e) => setNewCategory(e.target.value)}
                />
              )}
              <div className="flex items-center gap-2 mt-2">
                <input
                  type="checkbox"
                  id="editNewCat"
                  checked={useNewCategory}
                  onChange={(e) => setUseNewCategory(e.target.checked)}
                  className="rounded border-slate-300"
                />
                <label htmlFor="editNewCat" className="text-sm text-slate-600">Create new category</label>
              </div>
            </div>
            <div className="space-y-2">
              <Label>Model</Label>
              <Input
                placeholder="Enter model name"
                value={formData.model}
                onChange={(e) => setFormData({ ...formData, model: e.target.value })}
              />
            </div>
            <div className="space-y-2">
              <Label>Price (€)</Label>
              <Input
                type="number"
                step="0.01"
                placeholder="0.00"
                value={formData.price}
                onChange={(e) => setFormData({ ...formData, price: e.target.value })}
              />
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setIsEditModalOpen(false)}>
              <X className="w-4 h-4 mr-2" />
              Cancel
            </Button>
            <Button onClick={handleEdit} className="bg-blue-600 hover:bg-blue-700">
              <Save className="w-4 h-4 mr-2" />
              Save Changes
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Delete Confirmation Dialog */}
      <AlertDialog open={isDeleteDialogOpen} onOpenChange={setIsDeleteDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Are you sure?</AlertDialogTitle>
            <AlertDialogDescription>
              This will permanently delete <strong>{selectedProduct?.model}</strong> from the inventory.
              This action cannot be undone.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel onClick={() => setSelectedProduct(null)}>Cancel</AlertDialogCancel>
            <AlertDialogAction onClick={handleDelete} className="bg-red-600 hover:bg-red-700">
              <Trash2 className="w-4 h-4 mr-2" />
              Delete
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>

      {/* Credential Generator Modal */}
      <Dialog open={isCredentialModalOpen} onOpenChange={setIsCredentialModalOpen}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <Key className="w-5 h-5" />
              Customer Credentials Manager
            </DialogTitle>
            <DialogDescription>
              Generate or set weekly credentials for customers.
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 py-4">
            {/* Current Credentials Display */}
            <div className="bg-slate-50 p-4 rounded-lg">
              <p className="text-sm font-medium text-slate-700 mb-2">Current Customer Credentials:</p>
              <div className="space-y-1">
                <p className="text-sm"><span className="text-slate-500">Username:</span> <span className="font-mono font-semibold">{customerCredentials.username}</span></p>
                <p className="text-sm"><span className="text-slate-500">Password:</span> <span className="font-mono font-semibold">{customerCredentials.password}</span></p>
              </div>
            </div>

            {/* Generate Random */}
            <div className="border-t pt-4">
              <p className="text-sm font-medium text-slate-700 mb-2">Option 1: Auto-Generate</p>
              <Button 
                onClick={generateRandomCredentials}
                variant="outline"
                className="w-full"
              >
                <RefreshCw className="w-4 h-4 mr-2" />
                Generate Random Credentials
              </Button>
            </div>

            {/* Custom Credentials */}
            <div className="border-t pt-4">
              <p className="text-sm font-medium text-slate-700 mb-2">Option 2: Set Custom</p>
              <div className="space-y-2">
                <Input
                  placeholder="Enter username"
                  value={customUsername}
                  onChange={(e) => setCustomUsername(e.target.value)}
                />
                <Input
                  placeholder="Enter password"
                  value={customPassword}
                  onChange={(e) => setCustomPassword(e.target.value)}
                />
                <Button 
                  onClick={generateCustomCredentials}
                  variant="outline"
                  className="w-full"
                  disabled={!customUsername || !customPassword}
                >
                  <Key className="w-4 h-4 mr-2" />
                  Set Custom Credentials
                </Button>
              </div>
            </div>

            {/* Generated Preview */}
            {generatedCredentials && (
              <div className="bg-green-50 border border-green-200 p-4 rounded-lg">
                <p className="text-sm font-medium text-green-700 mb-2">New Credentials Preview:</p>
                <div className="space-y-1">
                  <p className="text-sm"><span className="text-green-600">Username:</span> <span className="font-mono font-semibold">{generatedCredentials.username}</span></p>
                  <p className="text-sm"><span className="text-green-600">Password:</span> <span className="font-mono font-semibold">{generatedCredentials.password}</span></p>
                </div>
                <Button 
                  onClick={copyToClipboard}
                  variant="outline"
                  size="sm"
                  className="mt-2 w-full"
                >
                  {copied ? <Check className="w-4 h-4 mr-2" /> : <Copy className="w-4 h-4 mr-2" />}
                  {copied ? 'Copied!' : 'Copy to Clipboard'}
                </Button>
              </div>
            )}
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={() => setIsCredentialModalOpen(false)}>
              <X className="w-4 h-4 mr-2" />
              Cancel
            </Button>
            <Button 
              onClick={saveCredentials} 
              className="bg-purple-600 hover:bg-purple-700"
              disabled={!generatedCredentials}
            >
              <Save className="w-4 h-4 mr-2" />
              Save Credentials
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Import CSV Modal */}
      <Dialog open={isImportModalOpen} onOpenChange={setIsImportModalOpen}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <Upload className="w-5 h-5" />
              Import CSV
            </DialogTitle>
            <DialogDescription>
              Upload a CSV file to add products to the inventory.
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 py-4">
            <div className="bg-slate-50 p-4 rounded-lg">
              <p className="text-sm font-medium text-slate-700 mb-2">CSV Format:</p>
              <p className="text-sm text-slate-600">Category,Models,Price (€)</p>
              <p className="text-xs text-slate-500 mt-1">Example: iPhone,15 Pro Max,999</p>
            </div>

            <Input
              type="file"
              accept=".csv"
              ref={fileInputRef}
              onChange={handleFileUpload}
              className="cursor-pointer"
            />

            {importError && (
              <div className="bg-red-50 border border-red-200 p-3 rounded-lg">
                <p className="text-sm text-red-600">{importError}</p>
              </div>
            )}

            {importSuccess && (
              <div className="bg-green-50 border border-green-200 p-3 rounded-lg">
                <p className="text-sm text-green-600">{importSuccess}</p>
              </div>
            )}
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={() => setIsImportModalOpen(false)}>
              <X className="w-4 h-4 mr-2" />
              Close
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
