const express = require('express');
const mysql = require('mysql2/promise');
const cors = require('cors');
require('dotenv').config();

const app = express();
const PORT = process.env.PORT || 3001;

// Middleware
app.use(cors());
app.use(express.json());

// Database connection pool
const pool = mysql.createPool({
  host: process.env.DB_HOST || 'localhost',
  user: process.env.DB_USER || 'root',
  password: process.env.DB_PASSWORD || '',
  database: process.env.DB_NAME || '39ricambi',
  waitForConnections: true,
  connectionLimit: 10,
  queueLimit: 0
});

// Test database connection
async function testConnection() {
  try {
    const connection = await pool.getConnection();
    console.log('✅ Database connected successfully');
    connection.release();
  } catch (error) {
    console.error('❌ Database connection failed:', error.message);
    process.exit(1);
  }
}

// ==================== CREDENTIALS API ====================

// Get current customer credentials
app.get('/api/credentials', async (req, res) => {
  try {
    const [rows] = await pool.execute(
      'SELECT username, password FROM customer_credentials ORDER BY id DESC LIMIT 1'
    );
    
    if (rows.length === 0) {
      // Return default if none exists
      return res.json({
        success: true,
        data: { username: 'customer', password: 'screen2024' }
      });
    }
    
    res.json({
      success: true,
      data: rows[0]
    });
  } catch (error) {
    console.error('Error fetching credentials:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch credentials'
    });
  }
});

// Update customer credentials (admin only)
app.post('/api/credentials', async (req, res) => {
  try {
    const { username, password, adminUsername, adminPassword } = req.body;
    
    // Verify admin credentials
    if (adminUsername !== process.env.ADMIN_USERNAME || 
        adminPassword !== process.env.ADMIN_PASSWORD) {
      return res.status(401).json({
        success: false,
        message: 'Unauthorized: Invalid admin credentials'
      });
    }
    
    // Validate input
    if (!username || !password) {
      return res.status(400).json({
        success: false,
        message: 'Username and password are required'
      });
    }
    
    // Update or insert credentials
    await pool.execute(
      `INSERT INTO customer_credentials (id, username, password) 
       VALUES (1, ?, ?) 
       ON DUPLICATE KEY UPDATE username = ?, password = ?`,
      [username, password, username, password]
    );
    
    res.json({
      success: true,
      message: 'Credentials updated successfully',
      data: { username, password }
    });
  } catch (error) {
    console.error('Error updating credentials:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update credentials'
    });
  }
});

// Login endpoint
app.post('/api/login', async (req, res) => {
  try {
    const { username, password } = req.body;
    
    // Check admin login
    if (username === process.env.ADMIN_USERNAME && 
        password === process.env.ADMIN_PASSWORD) {
      return res.json({
        success: true,
        role: 'admin',
        message: 'Admin login successful'
      });
    }
    
    // Check customer login from database
    const [rows] = await pool.execute(
      'SELECT username, password FROM customer_credentials ORDER BY id DESC LIMIT 1'
    );
    
    const customerCreds = rows[0] || { username: 'customer', password: 'screen2024' };
    
    if (username === customerCreds.username && password === customerCreds.password) {
      return res.json({
        success: true,
        role: 'user',
        message: 'Customer login successful'
      });
    }
    
    res.status(401).json({
      success: false,
      message: 'Invalid credentials'
    });
  } catch (error) {
    console.error('Login error:', error);
    res.status(500).json({
      success: false,
      message: 'Login failed'
    });
  }
});

// ==================== PRODUCTS API ====================

// Get all products
app.get('/api/products', async (req, res) => {
  try {
    const [rows] = await pool.execute(
      'SELECT id, category, model, price FROM products ORDER BY category, model'
    );
    
    res.json({
      success: true,
      data: rows
    });
  } catch (error) {
    console.error('Error fetching products:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch products'
    });
  }
});

// Add new product
app.post('/api/products', async (req, res) => {
  try {
    const { id, category, model, price, adminUsername, adminPassword } = req.body;
    
    // Verify admin
    if (adminUsername !== process.env.ADMIN_USERNAME || 
        adminPassword !== process.env.ADMIN_PASSWORD) {
      return res.status(401).json({
        success: false,
        message: 'Unauthorized'
      });
    }
    
    await pool.execute(
      'INSERT INTO products (id, category, model, price) VALUES (?, ?, ?, ?)',
      [id, category, model, price]
    );
    
    res.json({
      success: true,
      message: 'Product added successfully'
    });
  } catch (error) {
    console.error('Error adding product:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to add product'
    });
  }
});

// Update product
app.put('/api/products/:id', async (req, res) => {
  try {
    const { category, model, price, adminUsername, adminPassword } = req.body;
    const { id } = req.params;
    
    // Verify admin
    if (adminUsername !== process.env.ADMIN_USERNAME || 
        adminPassword !== process.env.ADMIN_PASSWORD) {
      return res.status(401).json({
        success: false,
        message: 'Unauthorized'
      });
    }
    
    await pool.execute(
      'UPDATE products SET category = ?, model = ?, price = ? WHERE id = ?',
      [category, model, price, id]
    );
    
    res.json({
      success: true,
      message: 'Product updated successfully'
    });
  } catch (error) {
    console.error('Error updating product:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update product'
    });
  }
});

// Delete product
app.delete('/api/products/:id', async (req, res) => {
  try {
    const { adminUsername, adminPassword } = req.body;
    const { id } = req.params;
    
    // Verify admin
    if (adminUsername !== process.env.ADMIN_USERNAME || 
        adminPassword !== process.env.ADMIN_PASSWORD) {
      return res.status(401).json({
        success: false,
        message: 'Unauthorized'
      });
    }
    
    await pool.execute('DELETE FROM products WHERE id = ?', [id]);
    
    res.json({
      success: true,
      message: 'Product deleted successfully'
    });
  } catch (error) {
    console.error('Error deleting product:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to delete product'
    });
  }
});

// Bulk import products
app.post('/api/products/import', async (req, res) => {
  try {
    const { products, adminUsername, adminPassword } = req.body;
    
    // Verify admin
    if (adminUsername !== process.env.ADMIN_USERNAME || 
        adminPassword !== process.env.ADMIN_PASSWORD) {
      return res.status(401).json({
        success: false,
        message: 'Unauthorized'
      });
    }
    
    // Insert products one by one
    let imported = 0;
    for (const product of products) {
      try {
        await pool.execute(
          'INSERT INTO products (id, category, model, price) VALUES (?, ?, ?, ?)',
          [product.id, product.category, product.model, product.price]
        );
        imported++;
      } catch (err) {
        // Skip duplicates
        console.log('Skipping duplicate:', product.id);
      }
    }
    
    res.json({
      success: true,
      message: `Imported ${imported} products`,
      imported
    });
  } catch (error) {
    console.error('Error importing products:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to import products'
    });
  }
});

// Health check endpoint
app.get('/api/health', async (req, res) => {
  try {
    await pool.execute('SELECT 1');
    res.json({ status: 'OK', database: 'Connected' });
  } catch (error) {
    res.status(500).json({ status: 'Error', database: 'Disconnected' });
  }
});

// Start server
app.listen(PORT, async () => {
  console.log(`🚀 Server running on port ${PORT}`);
  await testConnection();
});

module.exports = app;
